<?php namespace ImCatalog;

class Controller
{
	const CATEGORY = 1;

	const ITEM = 2;

	/**
	 * @var $processor - The processor instance
	 */
	public $processor;

	/**
	 * @var $trigger - The actions trigger
	 */
	public $trigger;

	/**
	 * @var $parser - ItemManager's template parser
	 */
	public $parser;

	/**
	 * @var $header - Catalog header output
	 */
	public $header;

	/**
	 * @var $breadcrumbs - Catalog breadcrumbs navi
	 */
	public $breadcrumbs;

	/**
	 * @var $searchform - Catalog search form output
	 */
	public $searchform;

	/**
	 * @var $content - Catalog content output
	 */
	public $content;

	/**
	 * @var $pagination - Catalog pagination output
	 */
	public $pagination;

	/**
	 * @var $title - Catalog current page title
	 */
	public $title;

	/**
	 * @var $footer - Catalog footer output
	 */
	public $footer;

	/**
	 * @var $pageid - The current page slug
	 */
	private $pageid;

	/**
	 * @var $router - The router
	 */
	public $router;

	/**
	 * @var $currentItem - The current item data
	 */
	private $currentItem;

	/**
	 * @var $currentRealCategory - The current real category data
	 */
	private $currentCategory;


	private $cachedOutput;

	/**
	 * Initializes some instances and includes librarys that we work with
	 *
	 * @param $processor - Instance of Processor class
	 */
	public function init(Processor $processor, ActionsTrigger $trigger)
	{
		if(!defined('IS_ADMIN_PANEL')) {
			include($processor->config->pluginRoot.'/tpl/frontend.php');
			!file_exists($processor->config->pluginRoot.'/tpl/custom.frontend.php')
				or include($processor->config->pluginRoot.'/tpl/custom.frontend.php');
		} else {
			include($processor->config->pluginRoot.'/tpl/admin.php');
			!file_exists($processor->config->pluginRoot.'/tpl/custom.admin.php')
				or include($processor->config->pluginRoot.'/tpl/custom.admin.php');
		}
		include_once(GSPLUGINPATH.'imanager/phpthumb/ThumbLib.inc.php');
		$this->imanager = imanager();
		$this->processor = $processor;
		$this->trigger = $trigger;
		$this->parser = $this->imanager->getTemplateEngine();
		\MsgReporter::$dir = $this->processor->config->pluginId;
		$this->pageid = (isset($_GET['id']) && !empty($_GET['id']) ? $_GET['id'] : 'index');
		$this->router = $this->processor->getRouter();
		$this->sectionCache = $this->imanager->getSectionCache();
	}


	public function setCurrentItem($categoryid, $itemid)
	{
		$this->currentItem = $this->processor->getItem($categoryid, $itemid);
		if($this->currentItem && $this->currentItem->active != 1) $this->currentItem = null;
		if(!empty($this->currentItem)) {
			$this->currentCategory = $this->processor->getItem($this->processor->config->dummyCategoryId,
				$this->currentItem->categoryid
			);
		}
	}


	/**
	 * This method renders the whole admin section
	 *
	 * @return string
	 */
	public function renderAdmin()
	{
		$output = $this->renderSection('AdminHeader');
		$output .= $this->renderSection('AdminBody');
		$output = $this->renderSection('AdminMessages', array($output));
		return $output;
	}


	/**
	 * Method for rendering the frontend contentent templates
	 *
	 * @return string
	 */
	public function renderFrontend()
	{
		if(!$this->router->catalogPage) {
			$this->__clean();
			return;
		}

		// Section cache enabled?
		if($this->processor->config->useSectionCache &&
			($this->cachedOutput = $this->sectionCache->get(
				str_replace('-', '',
					$this->imanager->sanitizer->pageName($this->processor->config->requestUri)),
						$this->processor->config->sectionCacheTime))) {
			if($this->processor->config->replaceContent) $this->buildFrontendCachedOutput();
		}

		global $title;

		$this->setCurrentItem($this->router->currentRoute['categoryid'], $this->router->currentRoute['id']);

		$this->title = $title;
		$this->searchform = '';
		$this->header = '';

		// Catalog start page only
		if($this->pageid == $this->processor->config->catalogStartPage &&
			!isset($_POST['search']) && !isset($_GET['search'])
		)
		{
			if($this->processor->config->displayCategoryListSearchForm)
			{
				$this->searchform = $this->renderSection('FrontendSearchForm');
			}

			if($this->processor->config->displayCategoryListHeader)
			{
				$this->header = $this->renderSection('FrontendCategoryListHeader');
			}

			$this->renderSection('FrontendCategoryContent');

		// Product list page
		} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::CATEGORY)
		{
			if(!$this->currentItem) { \Util::redirect('../'); exit(); }

			if($this->processor->config->displayItemListSearchForm)
			{
				$this->searchform = $this->renderSection('FrontendSearchForm');
			}

			if($this->processor->config->displayItemListHeader)
			{
				$this->header = $this->renderSection('FrontendItemListHeader');
			}

			$this->renderSection('FrontendItemListContent');

			// Product details page
		} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::ITEM)
		{
			if(!$this->currentItem) { \Util::redirect('../'); exit(); }

			if($this->processor->config->displayItemDetailsSearchForm)
			{
				$this->searchform = $this->renderSection('FrontendSearchForm');
			}

			if($this->processor->config->displayItemDetailsHeader)
			{
				$this->header = $this->renderSection('FrontendItemDetailsHeader');
			}

			$this->renderSection('FrontendItemDetailsContent');

			// Search result page
		} elseif($this->pageid == $this->processor->config->searchResultPage)
		{
			if($this->processor->config->displaySearchResultSearchForm)
			{
				$this->searchform = $this->renderSection('FrontendSearchForm');
			}

			if($this->processor->config->displaySearchResultHeader)
			{
				$this->header = $this->renderSection('FrontendSearchResultHeader');
			}

			$this->renderSection('FrontendSearchResultContent');
		// None of this applies, just quite
		} else { return; }

		if($this->processor->config->displayBreadcrumbs)
		{
			$this->breadcrumbs = $this->renderSection('FrontendBreadCrumbs');
		}

		if($this->processor->config->debug)
		{
			$this->content = $this->renderSection('AdminMessages', array('[[messages]]' . $this->content));
		}


		if($this->processor->config->replaceContent) $this->buildFrontendOutput();
	}


	/**
	 *
	 * @param $sectionName
	 * @param array $params
	 *
	 * @return mixed
	 */
	public function renderSection($sectionName, $params = array())
	{
		$method = '__render'.$sectionName;
		return !empty($params) ? $this->{$method}($params) : $this->{$method}();
	}


	/**
	 * Renders the admin header section:
	 *
	 * <div id="manager-header">
	 *      <h3 class="floated">...</h3>
	 *      ...
	 * </div>
	 *
	 *  @return string
	 */
	private function __renderAdminHeader()
	{
		return $this->parser->render($this->overviewHeader, array(
				'plugin_name' => Util::i18n_r('plugin_name'),
				'plugin_id' => $this->processor->config->pluginId,
				'current_1' => !isset($_GET['section']) ? 'class="current"' : '',
				'navi_label1' => Util::i18n_r('navi_label1'),
				'current_2' => (isset($_GET['section']) && $_GET['section'] == 'items') ? 'class="current"' : '',
				'navi_label2' => Util::i18n_r('navi_label2'),
				'current_3' => (isset($_GET['section']) && $_GET['section'] == 'categories') ? 'class="current"' : '',
				'navi_label3' => Util::i18n_r('navi_label3'),
			)
		);
	}


	/**
	 * Method for rendering the breadcrumb navi
	 *
	 * @return string
	 */
	private function __renderFrontendBreadCrumbs()
	{
		global $url;
		$crump_tpl = '';
		if($this->pageid == $this->processor->config->catalogStartPage &&
			!isset($_POST['search']) && !isset($_Get['search']))
		{
			$crump_tpl = $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL.$this->processor->config->trailingSlash : './',
					'class' => ' class="active"',
					'crumb_label' => $this->title
				)
			);
		} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::CATEGORY) {

			$crump_tpl = $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL.$this->processor->config->trailingSlash : '../',
					'class' => '',
					'crumb_label' => $this->title
				)
			);
			$crump_tpl .= $this->parser->render($this->crumb, array(
					'href' => '',
					'class' => ' class="active"',
					'crumb_label' => $this->currentItem->name
				)
			);
		// Product details page
		} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::ITEM)
		{
			$crump_tpl = $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL.$this->processor->config->trailingSlash : '../../',
					'class' => '',
					'crumb_label' => $this->title
				)
			);
			$crump_tpl .= $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL : '../',
					'class' => '',
					'crumb_label' => $this->currentCategory->name
				)
			);
			$crump_tpl .= $this->parser->render($this->crumb, array(
					'href' => './',
					'class' => ' class="active"',
					'crumb_label' => $this->currentItem->name
				)
			);
		} elseif($this->pageid == $this->processor->config->searchResultPage &&
			(isset($_POST['search']) || isset($_GET['search'])))
		{
			$data_index = getXml(GSDATAPAGESPATH . $this->processor->config->catalogStartPage . '.xml');
			$crump_tpl = $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL.$this->processor->config->trailingSlash : '../',
					'class' => '',
					'crumb_label' => $data_index->title
				)
			);
			$crump_tpl .= $this->parser->render($this->crumb, array(
					'href' => ($url == 'index') ? IM_SITE_URL.$this->processor->config->trailingSlash : './',
					'class' => ' class="active"',
					'crumb_label' => $this->title
				)
			);
		}

		return $this->parser->render($this->breadCrumbs, array(
				'crumbs' => $crump_tpl
			)
		);
	}


	/**
	 * Rendering for the search form
	 *
	 * @return string
	 */
	private function __renderFrontendSearchForm()
	{
		return $this->parser->render($this->serchForm, array(
				'search_action' => ($this->processor->config->searchResultPage != 'index')
					? $this->processor->config->searchResultPageUri : $this->processor->config->trailingSlash,
				'placeholder' => Util::i18n_r('search_field_placeholder')
			)
		);
	}


	/**
	 * Renders the frontend category list header section:
	 *
	 * <div class="sc-cl-header">
	 *     <h1>...</h1>
	 *      ...
	 * </div>
	 *
	 *  @return string
	 */
	private function __renderFrontendCategoryListHeader()
	{
		return $this->parser->render($this->categoryListHeader, array(
				'category_list_title' => Util::i18n_r('category_list_title'),
				'search_form' => $this->searchform
			)
		);
	}

	/**
	 * Renders the frontend item list header section:
	 *
	 * <div class="sc-il-header">
	 *     <h1>...</h1>
	 *      ...
	 * </div>
	 *
	 *  @return string
	 */
	private function __renderFrontendItemListHeader()
	{
		return $this->parser->render($this->itemListHeader, array(
				'item_list_title' => Util::i18n_r('item_list_title'),
				'search_form' => $this->searchform
			)
		);
	}


	/**
	 * Renders the frontend item list header section:
	 *
	 * <div class="sc-il-header">
	 *     <h1>...</h1>
	 *      ...
	 * </div>
	 *
	 *  @return string
	 */
	private function __renderFrontendItemDetailsHeader()
	{
		return $this->parser->render($this->itemDetailsHeader, array(
				'item_details_title' => Util::i18n_r('item_details_title'),
				'search_form' => $this->searchform
			)
		);
	}


	/**
	 * Renders the search results header section:
	 *
	 * <div class="sc-il-header">
	 *     <h1>...</h1>
	 *      ...
	 * </div>
	 *
	 *  @return string
	 */
	private function __renderFrontendSearchResultHeader()
	{
		return $this->parser->render($this->searchResultHeader, array(
				'search_result_title' => Util::i18n_r('search_result_title'),
				'search_form' => $this->searchform
			)
		);
	}


	/**
	 * Renders errors/notifications
	 */
	private function __renderAdminMessages($arg)
	{
		if(\MsgReporter::msgs())
		{
			return $this->parser->render($arg[0], array(
					'messages' => \MsgReporter::buildMsg())
			);
		} else
		{
			return $this->parser->render($arg[0], array(
					'messages' => '')
			);
		}
	}


	/**
	 * Renders body of the admin section
	 *
	 * <div id="itemContent" class="manager-wrapper">...</div>
	 *
	 * @return string - A rendered sections template
	 */
	private function __renderAdminBody()
	{
		// Categories section
		if(isset($_GET['section']) && $_GET['section'] == 'categories') {
			if(isset($_GET['action']) && $_GET['action'] == 'edit') {
				return $this->renderSection('AdminCategoryEditor');
			} else {
				return $this->renderSection('AdminCategoryList');
			}
		}
		// Items section
		elseif(isset($_GET['section']) && $_GET['section'] == 'items') {
			if(isset($_GET['action']) && $_GET['action'] == 'edit') {
				return $this->renderSection('AdminItemEditor');
			} else {
				return $this->renderSection('AdminItemList');
			}
		}
		// Dashboard
		else {
			return $this->renderSection('AdminDashboard');
		}
	}


	private function __renderFrontendCategoryContent() {
		return $this->renderSection('FrontendCategoryList');
	}


	private function __renderFrontendItemListContent() {
		return $this->renderSection('FrontendItemList');
	}


	private function __renderFrontendItemDetailsContent() {
		return $this->renderSection('FrontendItemDetails');
	}


	/**
	 * This method renders category list view
	 */
	private function __renderFrontendCategoryList()
	{
		$rows = '';
		$params = array(
			'section' => 'frontend',
			'selector' => 'active=1',
			'pageurl' => $this->processor->config->pageNumberParam
		);
		$result = $this->processor->getCategories($params);

		if(isset($result['total']) && $result['total'] > 0 && !empty($result['categories']))
		{
			foreach($result['categories'] as $category)
			{
				if($this->processor->config->categoryImageResize)
				{
					$path = Util::getResizedUrl($category,
						$this->processor->config->categoryImageIndex,
						$this->processor->config->categoryImageMaxWidth,
						$this->processor->config->categoryImageMaxHeight,
						'resize'
					);
				} else { $path = IM_SITE_URL.$category->image[$this->processor->config->categoryImageIndex]; }

				if(empty($path))
				{
					$path = IM_SITE_URL . 'plugins/' . $this->processor->config->pluginId . '/images/thumb300x300.png';
				}

				$replace = array();
				foreach($category as $name => $value) {
					if($name == 'image') {
						$replace[$name] = $path;
						continue;
					}
					$replace[$name] = ($name == 'image_title') ? $value[0] : $value;
				}
				$replace['href'] = './'.$category->slug.$this->processor->config->trailingSlash;
				$replace['info_link'] = Util::i18n_r('categorylist_infolink_label');
				$replace['category_info'] =  $this->parser->render($this->categoryInfo, $replace);
				$rows .= $this->parser->render($this->categoryRow, $replace);
			}
			$this->content = $rows;
			$this->pagination = $result['pagination'];
			$this->total = $result['total'];
		}
	}

	/**
	 * This method renders frontend item list rows
	 */
	private function __renderFrontendItemList($args = null)
	{
		$rows = '';
		$tag_selector = '';
		$tag_url = '';
		if(!empty($_GET['tag'])) {
			$tag_selector = '&&tags=%'.$this->imanager->sanitizer->text($_GET['tag']).'%';
			$tag_url = '?tag='.$this->imanager->sanitizer->text($_GET['tag']);
		}

		$params = array(
			'section' => 'frontend',
			'selector' => 'active=1'.$tag_selector,
			'pageurl' => $this->processor->config->pageNumberParam,
			'tagurl' => $tag_url
		);
		if($args['routeid']) $this->router->currentRoute['id'] = (int) $args['routeid'];
		$result = $this->processor->getItems($this->router->currentRoute['id'], $params);

		if(isset($result['total']) && $result['total'] > 0 && !empty($result['items']))
		{
			foreach($result['items'] as $item)
			{
				if($this->processor->config->itemListImageResize)
				{
					$path = Util::getResizedUrl($item,
						$this->processor->config->itemListImageIndex,
						$this->processor->config->itemListImageMaxWidth,
						$this->processor->config->itemListImageMaxHeight,
						'resize'
					);
				} else { $path = $item->image[$this->processor->config->itemListImageIndex]; }

				$replace = array();
				foreach($item as $name => $value) {
					if($name == 'image') {
						$replace[$name] = $path;
						continue;
					}
					$replace[$name] = ($name == 'image_title') ? $value[0] : $value;
				}
				$replace['href'] = './'.$item->slug.$this->processor->config->trailingSlash;

				$replace['figure'] = '';
				if(isset($replace['image']))
				{
					$replace['figure'] = $this->parser->render($this->itemFigure, array(
							'href' => $replace['href'],
							'image' => $replace['image'],
							'image_title' => $replace['image_title'],
						)
					);
					// delete unwanted stuff
					unset($replace['image']);
					unset($replace['image_title']);
				}

				$tags_wrapper_tpl = '';
				if($this->processor->config->displayItemListTags && !empty($item->tags))
				{
					$tags_tpl = '';
					$tags = array_map('trim', explode(',', $item->tags));
					if(is_array($tags)) {
						foreach($tags as $tag)
						{
							$tags_tpl .= $this->parser->render($this->tag, array(
									'href' => '?tag='.urlencode(strtolower($tag)),
									'tag_label' => $tag
								)
							);
						}
						$tags_wrapper_tpl = $this->parser->render($this->tagsWrapper, array('tags' =>  $tags_tpl));
					}
				}

				$replace['tags'] = $tags_wrapper_tpl;

				$replace['item_info'] = $this->parser->render($this->itemInfo, array(
						'href' => $replace['href'],
						'info_link' => Util::i18n_r('itemlist_infolink_label')
					)
				);

				$rows .= $this->parser->render($this->itemRow, $replace);
			}
			$this->content = $rows;
			$this->pagination = $result['pagination'];
			$this->total = $result['total'];
		}
	}


	private function __renderFrontendItemDetails()
	{
		if(empty($this->currentItem)) {
			$this->content = $this->parser->render($this->itemDetailsContent, array(
					'description' => Util::i18n_r('err_item_not_found')
				)
			);
			return;
		}
		$output = $this->itemDetailsContent;
		if(!empty($this->currentItem->image))
		{
			$path_r = array();
			$path = array();
			foreach($this->currentItem->image as $key => $image)
			{
				if($this->processor->config->itemDetailsImageResize)
				{
					$path_r[] = Util::getResizedUrl($this->currentItem,
						$key,
						$this->processor->config->itemDetailsImageSizes[$key]['w'],
						$this->processor->config->itemDetailsImageSizes[$key]['h'],
						'resize'
					);
				}

				$output = $this->parser->render($output, array(
						'image_'.$key => IM_SITE_URL.$image,
						'image_title_'.$key => $image,
						'image_s_'.$key => $path_r[$key],
						'image_title_s_'.$key => $path_r[$key]
					)
				);
			}
		}

		foreach($this->currentItem as $key => $value)
		{
			if(is_array($value)) continue;
			$output = $this->parser->render($output, array($key => $value));
		}

		$this->content = $output;
	}


	/**
	 * Method for rendering search results in the frontend
	 */
	private function __renderFrontendSearchResultContent()
	{
		$categories = $this->processor->getAllCategories();
		if(empty($categories)) {
			$this->content = Util::i18n_r('err_no_items_tosearch');
			return;
		}
		$index = (!empty($_POST['search']) ? $_POST['search'] : @$_GET['search']);

		$results = array();
		$searches = array_map('trim', explode(' ', $index, 3));

		foreach($categories as $category)
		{
			$f_results = array();
			foreach($this->processor->config->searchableFields as $sf)
			{
				$s_results = array();
				foreach($searches as $search)
				{
					if(empty(trim($search))) continue;

					$params = array(
						'section' => 'frontend',
						'selector' => 'active=1&&'.$sf.'=%'.$this->imanager->sanitizer->text($search).'%',
						'perpage' => 100//$this->processor->config->searchResultsPerPage
					);
					$result = $this->processor->getItems($category->id, $params);
					if($result['items']) {
						$s_results = $this->processor->itemsMerge($result['items']);
						$f_results = array_replace($f_results, $s_results);
					}
				}
			}
			if(!empty($f_results)) {
				$results[$category->id] = $f_results;
			}
		}
		// nothing was found
		if(empty($results))
		{
			$this->resultItems = 0;
			$this->resultCategories = count($categories);
			$this->content = Util::i18n_r('nothing_found');
			return;
		}

		// Let's loop through results
		$rows = '';
		$count = 0;
		$oldcat = null;
		foreach($results as $catid => $result)
		{
			$count += count($result);
			foreach($result as $item)
			{
				$output = $this->searchResultRow;
				$figure = '';
				if(!empty($item->image))
				{
					$path = '';
					foreach($item->image as $key => $image)
					{
						if($this->processor->config->searchResultImageResize)
						{
							$path = Util::getResizedUrl($item,
								$this->processor->config->searchResultImageIndex,
								$this->processor->config->searchResultImageMaxWidth,
								$this->processor->config->searchResultImageMaxHeight,
								'resize'
							);
						}

						$figure = $this->parser->render($this->itemFigure, array(
								'image' => $path,
								'image_title' => !empty($item->image_title[$key]) ? $item->image_title[$key] : '',
								'href' => '/'.$this->processor->config->catalogStartPage.'/'.$categories[$catid]->slug.
									'/'.$item->slug.$this->processor->config->trailingSlash
							)
						);
					}
				}

				foreach($item as $key => $value)
				{
					if(is_array($value)) continue;
					$replace = array($key => $value);
					if($this->processor->config->shortenSearchOutput) {
						$replace = array($key => (strlen($value) > $this->processor->config->searchOutputChars) ?
							substr($value, 0, $this->processor->config->searchOutputChars).'...' : $value);
					}
					$output = $this->parser->render($output, $replace);
				}

				$rows .= $this->parser->render($output, array(
						'category_title' => (($oldcat != $catid) ? $this->parser->render($this->h3Header, array(
							'title' => $categories[$catid]->name)) : ''),
						'figure' => $figure,
						'href' => '/'.$this->processor->config->catalogStartPage.'/'.$categories[$catid]->slug.
							'/'.$item->slug.$this->processor->config->trailingSlash
					)
				);
				$oldcat = $catid;
			}
		}
		$this->resultItems = $count;
		$this->resultCategories = count($categories);
		$this->content = $rows;
	}


	/**
	 * This method renders frontend item list rows
	 */
	private function __renderFrontendItemsSearch($cateroryid)
	{
		$rows = '';
		$params = array(
			'section' => 'frontend',
			'selector' => 'active=1',
			'pageurl' => $this->processor->config->pageNumberParam
		);

		$result = $this->processor->getItems($cateroryid, $params);

		if(isset($result['total']) && $result['total'] > 0 && !empty($result['items']))
		{
			foreach($result['items'] as $item)
			{
				if($this->processor->config->itemListImageResize)
				{
					$path = Util::getResizedUrl($item,
						$this->processor->config->itemListImageIndex,
						$this->processor->config->itemListImageMaxWidth,
						$this->processor->config->itemListImageMaxHeight,
						'resize'
					);
				} else { $path = $item->image[$this->processor->config->itemListImageIndex]; }

				$replace = array();
				foreach($item as $name => $value) {
					if($name == 'image') {
						$replace[$name] = $path;
						continue;
					}
					$replace[$name] = ($name == 'image_title') ? $value[0] : $value;
				}
				$replace['href'] = './'.$item->slug.$this->processor->config->trailingSlash;

				$replace['figure'] = '';
				if(isset($replace['image']))
				{
					$replace['figure'] = $this->parser->render($this->itemFigure, array(
							'href' => $replace['href'],
							'image' => $replace['image'],
							'image_title' => $replace['image_title'],
						)
					);
					// delete unwanted stuff
					unset($replace['image']);
					unset($replace['image_title']);
				}

				$tags_wrapper_tpl = '';
				if($this->processor->config->displayItemListTags && !empty($item->tags))
				{
					$tags_tpl = '';
					$tags = array_map('trim', explode(',', $item->tags));
					if(is_array($tags)) {
						foreach($tags as $tag)
						{
							$tags_tpl .= $this->parser->render($this->tag, array(
									'href' => '?tag='.urlencode(strtolower($tag)),
									'tag_label' => $tag
								)
							);
						}
						$tags_wrapper_tpl = $this->parser->render($this->tagsWrapper, array('tags' =>  $tags_tpl));
					}
				}

				$replace['tags'] = $tags_wrapper_tpl;

				$replace['item_info'] = $this->parser->render($this->itemInfo, array(
						'href' => $replace['href'],
						'info_link' => Util::i18n_r('itemlist_infolink_label')
					)
				);

				$rows .= $this->parser->render($this->itemRow, $replace);
			}
			$this->content = $rows;
			$this->pagination = $result['pagination'];
		}
	}


	/**
	 * An Ajax-called method
	 * This method renders the category list view only
	 *
	 * @return string
	 */
	public function renderAdminCategoryListRows()
	{
		$rows = '';
		$result = $this->processor->getCategories();

		if(isset($result['total']) && $result['total'] > 0 && !empty($result['categories']))
		{
			foreach($result['categories'] as $key => $category)
			{
				$path = Util::getResizedUrl($category, 0, 40, 40);
				if(empty($path))
				{
					$path = IM_SITE_URL.'plugins/'.$this->processor->config->pluginId.'/images/thumb40x40.png';
				}
				$rows .= $this->parser->render($this->categoryRow, array(
						'cover_image' => $path,
						'pos' => $category->position,
						'item_id' => $category->id,
						'plugin_id' => $this->processor->config->pluginId,
						'page_number' => !empty($result['page']) ? $result['page'] : 1,
						'item_title' => (strlen($category->name) > 13) ?
							substr($category->name, 0, 15).'...' : $category->name,
						'active' => !empty((int)$category->active) ? 'fa-check-square-o' : 'fa-square-o',
						'created' => date((string) $this->processor->config->timeFormat, (int)$category->created)
					)
				);
			}
		}
		return json_encode(array(
				'rows' => $rows,
				'pagination' => (!empty($result['pagination']) ? $result['pagination'] : '')
			)
		);
	}


	/**
	 * An Ajax-called method
	 * This method renders the item list view only
	 *
	 * @return string
	 */
	public function renderAdminItemListRows()
	{
		$currentCategoryId = $this->processor->getCurrentCategoryId();
		if(!$currentCategoryId) return;

		$rows = '';
		$result = $this->processor->getItems($currentCategoryId);
		if(isset($result['total']) && $result['total'] > 0 && !empty($result['items']))
		{
			foreach($result['items'] as $key => $item)
			{
				$path = Util::getResizedUrl($item, 0, 40, 40);
				if(empty($path))
				{
					$gsadmindir = (defined('GSADMIN') ? GSADMIN : 'admin');
					$path = IM_SITE_URL.'plugins/'.$this->processor->config->pluginId.'/images/thumb40x40.png';
				}
				$rows .= $this->parser->render($this->itemRow, array(
						'cover_image' => $path,
						'pos' => $item->position,
						'item_id' => $item->id,
						'plugin_id' => $this->processor->config->pluginId,
						'page_number' => !empty($result['page']) ? $result['page'] : 1,
						'item_title' => (strlen($item->name) > 13) ?
							substr($item->name, 0, 15).'...' : $item->name,
						'active' => !empty((int)$item->active) ? 'fa-check-square-o' : 'fa-square-o',
						'created' => date((string) $this->processor->config->timeFormat, (int)$item->created)
					)
				);
			}
		}
		return json_encode(array(
				'rows' => $rows,
				'pagination' => (!empty($result['pagination']) ? $result['pagination'] : '')
			)
		);
	}


	private function __renderAdminDashboard()
	{
		if(($data = $this->processor->getDashboardData()) === false) return;
		if(!empty($data->showSlider)) include(dirname(__DIR__).'/slider/lib/bootstrap.php');

		$recentRows = '';
		for($i = 0; $i < 3; $i++)
		{
			if(!empty($data->recentItems[$i]['src']))
			{
				$image = $this->parser->render($this->imageElement, array(
						'alt' => '',
						'src' => IM_SITE_URL.$data->recentItems[$i]['src'],
					)
				);
			} else
			{
				$image = $this->parser->render($this->imageElement, array(
						'alt' => '',
						'src' => IM_SITE_URL.'plugins/'.$this->processor->config->pluginId.'/images/thumb350x200.png',
					)
				);
			}
			$recentRows .= $this->parser->render($this->recentItemRow, array(
					'url' => (!empty($data->recentItems[$i]['id']) ? 'load.php?id='.$this->processor->config->pluginId.
						'&section=items&action=edit&edit='.$data->recentItems[$i]['id'] : '#'),
					'image' => $image,
					'title' => (!empty($data->recentItems[$i]['title']) ? $data->recentItems[$i]['title'] :
						Util::i18n_r('empty_recent_title')),
				)
			);
		}

		return $this->parser->render($this->dashboardWrapper, array(
				'items_total' => Util::i18n_r('items_total_label'),
				'it_total_number' => $data->itemsTotal,
				'categories_total' => Util::i18n_r('categories_total_label'),
				'ct_total_number' => $data->categoriesTotal,
				'recent_items' => Util::i18n_r('recent_items'),
				'recent-items' => $recentRows,
				'slider' => (!empty($data->showSlider) ? $slider->renderSlider() : ''),
				'creator' => (!empty($data->createdBy) ? $data->createdBy : ''),
				'creator_mail' => (!empty($data->createrMail) ? $data->createrMail : ''),
				'creator_phone' => (!empty($data->createrPhone) ? $data->createrPhone : ''),
				'infos' => (!empty($data->helpUrl) ? $data->helpUrl : '')

			)
		);

	}


	/**
	 * This method renders the category list view in the admin area
	 *
	 * @return string
	 */
	private function __renderAdminCategoryList()
	{
		$rows = '';
		$result = $this->processor->getCategories();
		if(isset($result['total']) && $result['total'] > 0 && !empty($result['categories']))
		{
			foreach($result['categories'] as $key => $category)
			{
				$path = Util::getResizedUrl($category, 0, 40, 40);
				if(empty($path))
				{
					$gsadmindir = (defined('GSADMIN') ? GSADMIN : 'admin');
					$path = IM_SITE_URL.'plugins/'.$this->processor->config->pluginId.'/images/thumb40x40.png';
				}
				$rows .= $this->parser->render($this->categoryRow, array(
						'cover_image' => $path,
						'pos' => $category->position,
						'item_id' => $category->id,
						'plugin_id' => $this->processor->config->pluginId,
						'page_number' => !empty($result['page']) ? $result['page'] : 1,
						'item_title' => (strlen($category->name) > 13) ?
							substr($category->name, 0, 15).'...' : $category->name,
						'active' => !empty((int)$category->active) ? 'fa-check-square-o' : 'fa-square-o',
						'created' => date((string) $this->processor->config->timeFormat, (int)$category->created)
					)
				);
			}
		}

		return $this->parser->render($this->categoryListForm, array(
				'table_position_label' => Util::i18n_r('table_position_label'),
				'table_name_label' => Util::i18n_r('table_name_label'),
				'table_created_label' => Util::i18n_r('table_created_label'),
				'table_active_label' => Util::i18n_r('table_active_label'),
				'item_rows' => $rows,
				'pagination' => (!empty($result['pagination']) ? $result['pagination'] : ''),
				'product_categories' => Util::i18n_r('product_categories'),
				'count' => (!empty($result['total']) ? $result['total'] : 0),
				'create_category' => Util::i18n_r('create_category_label'),
				'plugin_id' => $this->processor->config->pluginId,
				'page_number' => !empty($result['page']) ? $result['page'] : 1
			)
		);
	}


	/**
	 * This method renders the item list view in the admin area
	 *
	 * @return string
	 */
	private function __renderAdminItemList()
	{
		$currentCategoryId = $this->processor->getCurrentCategoryId();
		if(!$currentCategoryId) return;

		$rows = '';
		$result = $this->processor->getItems($currentCategoryId);

		if(isset($result['total']) && $result['total'] > 0 && !empty($result['items']))
		{
			foreach($result['items'] as $key => $item)
			{
				$path = Util::getResizedUrl($item, 0, 40, 40);
				if(empty($path))
				{
					$gsadmindir = (defined('GSADMIN') ? GSADMIN : 'admin');
					$path = IM_SITE_URL.'plugins/'.$this->processor->config->pluginId.'/images/thumb40x40.png';
				}
				$rows .= $this->parser->render($this->itemRow, array(
						'cover_image' => $path,
						'pos' => $item->position,
						'item_id' => $item->id,
						'plugin_id' => $this->processor->config->pluginId,
						'page_number' => !empty($result['page']) ? $result['page'] : 1,
						'item_title' => (strlen($item->name) > 13) ?
							substr($item->name, 0, 15).'...' : $item->name,
						'active' => !empty((int)$item->active) ? 'fa-check-square-o' : 'fa-square-o',
						'created' => date((string) $this->processor->config->timeFormat, (int)$item->created)
					)
				);
			}
		}

		$options = $this->renderCategoryOptions($currentCategoryId);

		return $this->parser->render($this->itemListForm, array(
				'select_category_label' => Util::i18n_r('select_category_label'),
				'options' => $options,
				'table_position_label' => Util::i18n_r('table_position_label'),
				'table_name_label' => Util::i18n_r('table_name_label'),
				'table_created_label' => Util::i18n_r('table_created_label'),
				'table_active_label' => Util::i18n_r('table_active_label'),
				'item_rows' => $rows,
				'pagination' => (!empty($result['pagination']) ? $result['pagination'] : ''),
				'product_categories' => Util::i18n_r('product_categories'),
				'count' => (!empty($result['total']) ? $result['total'] : 0),
				'create_item' => Util::i18n_r('create_item_label'),
				'plugin_id' => $this->processor->config->pluginId,
				'page_number' => !empty($result['page']) ? $result['page'] : 1
			)
		);
	}


	/**
	 * Renders the category selector options
	 *     <option value="...">...</option>
	 *     ...
	 * @return string
	 */
	private function renderCategoryOptions($currentCategoryId)
	{
		$categories = $this->processor->getAllCategories();
		$output = '';

		if(!empty($categories))
		{
			foreach($categories as $category)
			{
				$output .= $this->parser->render($this->selectOption, array(
					'value' => (int)$category->id,
					'selected' => (($currentCategoryId == $category->id) ? ' selected="selected"' : ''),
					'label' => $category->name
					)
				);
			}
		}
		return $output;
	}


	/**
	 * This method renders item data within category edit section
	 *
	 * @return bool|string - Rendered templete of the admin editor
	 */
	private function __renderAdminCategoryEditor()
	{
		$curitem = $this->processor->getItemForEdit($this->processor->config->dummyCategoryId,
			!empty($_GET['edit']) ? $_GET['edit'] : null);
		if(!$curitem) {
			\MsgReporter::setClause('err_item_select', array(), true);
			return false;
		}

		$timestamp = !empty($_POST['timestamp']) ? (int)$_POST['timestamp'] : time();

		return $this->parser->render($this->categoryEdotorWrapper, array(
				'plugin_id' => $this->processor->config->pluginId,
				'url_id' => (!empty($curitem->id) ? '&edit='.$curitem->id : ''),
				'page_number' => (!empty($_GET['page']) ? (int) $_GET['page'] : 1),
				'category_header' => (!empty($curitem->id) ? Util::i18n_r('edit_category_header') :
					Util::i18n_r('create_category_header')),
				'cat_id_label' => Util::i18n_r('cat_id_label'),
				'cat_id' => !empty($curitem->id) ? $curitem->id : Util::i18n_r('empty_current_item_id'),
				'cat_name_label' => Util::i18n_r('cat_name_label'),
				'cat_name_value' => !empty($curitem->name) ? $curitem->name : '',
				'cat_pos_label' => Util::i18n_r('cat_pos_label'),
				'cat_pos_value' => !empty($curitem->position) ? $curitem->position : (!empty($curitem->id) ? $curitem->id : ''),
				'cat_created_label' => Util::i18n_r('cat_created_label'),
				'cat_created_value' => !empty($curitem->created) ?
					(date((string) $this->processor->config->timeFormat, (int)$curitem->created)) : '',
				'cat_modified_label' => Util::i18n_r('cat_modified_label'),
				'cat_modified_value' => !empty($curitem->updated) ?
					(date((string) $this->processor->config->timeFormat, (int)$curitem->updated)) :
					Util::i18n_r('empty_modified_item_date'),
				'fields' => $this->getItemFieldsEditor($this->processor->config->dummyCategoryId, $curitem, $timestamp),
				'categoryid'  => $this->processor->config->dummyCategoryId,
				'timestamp' => $timestamp,
				'cat_save_button' => Util::i18n_r('cat_save_button')
			), true, array()
		);
	}


	/**
	 * This method renders item fields within item edit section
	 *
	 * @return bool|string - Rendered templete for the admin editor
	 */
	private function __renderAdminItemEditor()
	{
		$curcategoryid = $this->processor->getCurrentCategoryId();
		$curitem = $this->processor->getItemForEdit($curcategoryid,
			!empty($_GET['edit']) ? $_GET['edit'] : null);
		if(!$curitem) {
			\MsgReporter::setClause('err_item_select', array(), true);
			return false;
		}

		$timestamp = !empty($_POST['timestamp']) ? (int)$_POST['timestamp'] : time();

		return $this->parser->render($this->itemEdotorWrapper, array(
				'plugin_id' => $this->processor->config->pluginId,
				'url_id' => (!empty($curitem->id) ? '&edit='.$curitem->id : ''),
				'page_number' => (!empty($_GET['page']) ? (int) $_GET['page'] : 1),
				'item_header' => (!empty($curitem->id) ? Util::i18n_r('edit_item_header') :
					Util::i18n_r('create_item_header')),
				'item_id_label' => Util::i18n_r('item_id_label'),
				'item_id' => !empty($curitem->id) ? $curitem->id : Util::i18n_r('empty_current_item_id'),
				'item_name_label' => Util::i18n_r('item_name_label'),
				'item_name_value' => !empty($curitem->name) ? $curitem->name : '',
				'item_pos_label' => Util::i18n_r('item_pos_label'),
				'item_pos_value' => !empty($curitem->position) ? $curitem->position : (!empty($curitem->id) ? $curitem->id : ''),
				'item_created_label' => Util::i18n_r('item_created_label'),
				'item_created_value' => !empty($curitem->created) ?
					(date((string) $this->processor->config->timeFormat, (int)$curitem->created)) : '',
				'item_modified_label' => Util::i18n_r('item_modified_label'),
				'item_modified_value' => !empty($curitem->updated) ?
					(date((string) $this->processor->config->timeFormat, (int)$curitem->updated)) :
					Util::i18n_r('empty_modified_item_date'),
				'fields' => $this->getItemFieldsEditor($curcategoryid, $curitem, $timestamp),
				'categoryid'  => $curcategoryid,
				'timestamp' => $timestamp,
				'item_save_button' => Util::i18n_r('item_save_button')
			), true, array()
		);
	}


	/**
	 *	Renders item / categiry fields
	 *
	 * @return string - Rendered templete for the item fields
	 */
	private function getItemFieldsEditor($catid, \Item $curitem, $timestamp)
	{
		is_numeric($catid) or exit();
		// Get the item field templates of ItemManager
		$this->parser->init();
		$itemeditor = $this->parser->getTemplates('itemeditor');
		$fieldarea = $this->parser->getTemplate('fieldarea', $itemeditor);
		$infotext = $this->parser->getTemplate('infotext', $itemeditor);
		$required = $this->parser->getTemplate('required', $itemeditor);

		// Initialize fields
		$fc = new \FieldMapper();
		$fc->init($catid);
		$fields = $fc->filterFields('position', 'asc');
		$tplfields = Util::i18n_r('empty_fields');
		if($fields)
		{
			$tplfields = '';
			foreach($fields as $fieldname => $field)
			{
				// Input
				$inputClassName = 'Input'.ucfirst($field->type);
				$InputType = new $inputClassName($fields[$fieldname]);
				$output = $InputType->prepareOutput();

				// Field
				$fieldClassName = 'Field'.ucfirst($field->type);
				$fieldType = new $fieldClassName($this->parser);
				$fieldType->name = $fieldname;
				$fieldType->id = $fieldType->name;

				if(!empty($field->configs))
				{
					foreach($field->configs as $key => $val)
					{
						$fieldType->configs->$key = $val;
					}
				}

				// Hidden
				if($field->type == 'hidden') continue;
				// Dropdown
				if($field->type == 'dropdown') $fieldType->options = $field->options;
				// Image upload
				if($field->type == 'imageupload')
				{
					$fieldType->categoryid = $catid;
					$fieldType->itemid = $curitem->id;
					$fieldType->realid = $field->id;
					$fieldType->timestamp = $timestamp;
				}
				// file upload
				if($field->type == 'fileupload')
				{
					$fieldType->categoryid = $catid;
					$fieldType->itemid = $curitem->id;
					$fieldType->realid = $field->id;
					$fieldType->timestamp = $timestamp;
				}

				foreach($output as $outputkey => $outputvalue)
				{
					$fieldType->class = empty($field->fieldclass) ? $field->type.'-field' : $field->fieldclass;

					if(!empty($field->fieldcss)) $fieldType->style = $field->fieldcss;

					if(is_array($outputvalue))
					{
						$fieldType->{$outputkey} = array();
						$counter = 0;
						if(!isset($curitem->fields->{$fieldname}->{$outputkey})) continue;

						foreach($curitem->fields->{$fieldname}->{$outputkey} as $arrkey => $arrval)
						{
							$fieldType->{$outputkey}[] = (string) $curitem->fields->{$fieldname}->{$outputkey}[$counter];
							$counter++;
						}
					} else
					{
						$fieldType->{$outputkey} = !empty($curitem->fields->{$fieldname}->{$outputkey}) ?
							(string)$curitem->fields->{$fieldname}->{$outputkey} : '';
					}

					if(\MsgReporter::isError())
					{
						$fieldType->{$outputkey} = !empty($_POST[$fieldType->name]) ? $_POST[$fieldType->name] : '';
					}
				}

				// set default field values
				if(empty($fieldType->value) && !empty($field->default)) $fieldType->value = (string) $field->default;

				$tplinfotext = '';
				if(!empty($field->info))
				{
					$tplinfotext = $this->parser->render($infotext, array('infotext' => $field->info));
				}

				$tplrequired = '';
				if(!empty($field->required) && $field->required == 1) {
					$tplrequired = $this->parser->render($required, array());
				}

				if($field->type != 'chunk')
				{
					$tplfields .= $this->parser->render($fieldarea, array(
							'fieldid' =>  $field->name,
							'label' => $field->label,
							'infotext' => $tplinfotext,
							'area-class' => !empty($field->areaclass) ? $field->areaclass : 'fieldarea',
							'area-style' => !empty($field->areacss) ? ' style="'.$field->areacss.'"' : '',
							'label-class' => !empty($field->labelclass) ? $field->labelclass : '',
							'label-style' => !empty($field->labelcss) ? ' style="'.$field->labelcss.'"' : '',
							'required' => $tplrequired,
							'field' => $fieldType->render())
					);
				} else
				{
					$tplfields .= $fieldType->render();
				}
			}
		}
		return $tplfields;
	}

	/**
	 * Output builder
	 */
	protected function buildFrontendOutput()
	{
		global $title, $content, $metad, $metak;
		$uri = str_replace('-', '',
			$this->imanager->sanitizer->pageName($this->processor->config->requestUri)
		);
		if(!$this->processor->config->scatteredOutput)
		{
			if(($this->pageid == $this->processor->config->catalogStartPage) &&
				!isset($_POST['search']) && !isset($_GET['search']))
			{
				$title = !empty($this->title) ? $this->title : $title;

				$output = $this->parser->render($this->categoryListWrapper, array(
						'header' => $this->header,
						'breadcrumbs' => $this->breadcrumbs,
						'content' => $this->content,
						'pagination' => $this->pagination,
						'footer' => $this->footer
					)
				);
				if($this->processor->config->useSectionCache) {
					$this->sectionCache->save($output);
					if(!$value = $this->sectionCache->get($uri.'-title', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($title);
					}
				}

			// Product list page
			} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::CATEGORY)
			{
				$this->title = !empty($this->currentItem->metatitle) ?
					$this->currentItem->metatitle : $this->currentItem->name;
				$metad = !empty($this->currentItem->metadescription) ?
					$this->currentItem->metadescription :
					strip_tags(substr($this->currentItem->description, 0, 160)).'...';
				$metak = !empty($this->currentItem->metakeywords) ? $this->currentItem->metakeywords : '';

				$title = !empty($this->title) ? $this->title : $title;

				$output = $this->parser->render($this->itemListWrapper, array(
						'header' => $this->header,
						'breadcrumbs' => $this->breadcrumbs,
						'content' => $this->content,
						'pagination' => $this->pagination,
						'footer' => $this->footer
					)
				);
				if($this->processor->config->useSectionCache) {
					$this->sectionCache->save($output);
					if(!$value = $this->sectionCache->get($uri.'-title', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($title);
					}
					if(!$value = $this->sectionCache->get($uri.'-metad', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($metad);
					}
					if(!$value = $this->sectionCache->get($uri.'-metak', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($metak);
					}
				}

			// Product details page
			} elseif($this->router->currentRoute && $this->router->currentRoute['type'] == self::ITEM)
			{
				$this->title = !empty($this->currentItem->metatitle) ?
					$this->currentItem->metatitle : $this->currentItem->name;
				$title = !empty($this->title) ? $this->title : $title;
				$metad = !empty($this->currentItem->metadescription) ?
					$this->currentItem->metadescription :
					strip_tags(substr($this->currentItem->description, 0, 160)).'...';
				$metak = !empty($this->currentItem->metakeywords) ? $this->currentItem->metakeywords : '';

				$output = $this->parser->render($this->itemDetailsWrapper, array(
						'header' => $this->header,
						'breadcrumbs' => $this->breadcrumbs,
						'content' => $this->content,
						'footer' => $this->footer
					)
				);
				if($this->processor->config->useSectionCache) {
					$this->sectionCache->save($output);
					if(!$value = $this->sectionCache->get($uri.'-title', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($title);
					}
					if(!$value = $this->sectionCache->get($uri.'-metad', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($metad);
					}
					if(!$value = $this->sectionCache->get($uri.'-metak', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($metak);
					}
				}

			// Search result page
			} elseif($this->pageid == $this->processor->config->searchResultPage)
			{
				$title = (!empty($this->title) ? $this->title : $title);

				$resultinfo = '';
				if(!empty($this->resultItems)) {
					$resultinfo = \MsgReporter::getClause('search_results', array(
						'items' => $this->resultItems,
						'categories' => $this->resultCategories)
					);
				}
				$output = $this->parser->render($this->searchResultWrapper, array(
						'header' => $this->header,
						'breadcrumbs' => $this->breadcrumbs,
						'search_info' => $this->parser->render($this->searchInfo, array('info' => $resultinfo)),
						'content' => $this->content,
						'pagination' => $this->pagination,
						'footer' => $this->footer
					)
				);
				if($this->processor->config->useSectionCache) {
					$this->sectionCache->save($output);
					if(!$value = $this->sectionCache->get($uri.'-title', $this->processor->config->sectionCacheTime)) {
						$this->sectionCache->save($title);
					}
				}
			}
			$content = preg_replace('%&lt;p&gt;(.*?)catalog_content(.*?)&lt;/p&gt;%',
				'[[catalog_content]]', $content);
			// save output in section cache
			$content = $this->parser->render($content, array('catalog_content' => $output));
		}
	}

	/**
	 * Cached output builder
	 */
	protected function buildFrontendCachedOutput()
	{
		global $title, $content, $metad, $metak;
		$uri = str_replace('-', '',
			$this->imanager->sanitizer->pageName($this->processor->config->requestUri)
		);
		if($value = $this->sectionCache->get($uri.'-title', $this->processor->config->sectionCacheTime)) $title = $value;
		if($value = $this->sectionCache->get($uri.'-metad', $this->processor->config->sectionCacheTime)) $metad = $value;
		if($value = $this->sectionCache->get($uri.'-metak', $this->processor->config->sectionCacheTime)) $metak = $value;

		$content = preg_replace('%&lt;p&gt;(.*?)catalog_content(.*?)&lt;/p&gt;%',
			'[[catalog_content]]', $content);
		// save output in section cache
		$content = $this->parser->render($content, array('catalog_content' => $this->cachedOutput));
	}


	public function __clean()
	{
		global $content;
		$content = preg_replace('%&lt;p&gt;(.*?)catalog_content(.*?)&lt;/p&gt;%',
			'[[catalog_content]]', $content);
		$content = $this->parser->render($content, array('catalog_content' => ''));
	}

}