<?php namespace ImCatalog;

class Processor
{
	protected $imanager;
	protected $itemMapper;
	protected $dashboardData;
	protected $router;
	public $config;

	/**
	 * Processor constructor.
	 *
	 * @param $config - The SimpleCatalog Config object
	 */
	public function __construct(Config $config, Router $router)
	{
		$this->imanager = imanager();
		$this->itemMapper = $this->imanager->getItemMapper();
		$this->router = $router;
		$this->dashboardData = null;
		$this->config = $config;
	}

	/**
	 * Only when really necessary
	 *
	 * @return Router
	 */
	public function getRouter() {return $this->router;}


	/**
	 * This method returns an array of SC-categories
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	public function getCategories($args = array())
	{
		$condit = array(
			'type' =>     'simple_items',
			'section' =>  'admin',
			'order' =>    'DESC',
			'filter' =>   'position',
			'selector' => '',
			'pageurl' => 'load.php?id='.$this->config->pluginId.'&section=categories&page=',
		);
		$args = array_merge($condit, $args);

		if(!empty($args['type']) && $args['type'] == 'simple_items')
		{
			$perpage = (!empty($args['section']) && $args['section'] == 'admin') ?
				(int) $this->config->adminCatPerPage : (int) $this->config->frontendCatPerPage;
			$page = (int) str_replace('page', '', (!empty($_GET['page']) ? $_GET['page'] : 1));
			$next = ($page - 1) * $perpage + 1;
			$simpleItems = $this->getAllCategories($args['selector']);
			if(!$simpleItems) return array('categories' => $simpleItems, 'total' => 0, 'pagination' => '', 'page' => 1);
			$total = $this->itemMapper->countItems($simpleItems);
			$this->itemMapper->total = $total;
			$simpleItems = $this->itemMapper->filterSimpleItems($args['filter'], $args['order'], $next, $perpage, $simpleItems);
			$pagination = $this->itemMapper->pagination(array(
				'page' => $page,
				'pageurl' => $args['pageurl'],
				'limit' => $perpage)
			);

			return array('categories' => $simpleItems, 'total' => $total, 'pagination' => $pagination, 'page' => $page);
		}
	}


	/**
	 * This method returns an array of SC-items with certain category id
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	public function getItems($catid, $args = array())
	{
		$condit = array(
			'type' =>     'simple_items',
			'section' =>  'admin',
			'order' =>    'DESC',
			'filter' =>   'position',
			'selector' => '',
			'pageurl' => 'load.php?id='.$this->config->pluginId.'&section=items&page='
		);
		$args = array_merge($condit, $args);

		if(!empty($args['tagurl'])) {
			$args['pageurl'] = $args['tagurl'].str_replace('?', '&', $args['pageurl']);
		}

		if(!empty($args['type']) && $args['type'] == 'simple_items')
		{
			if(empty($args['perpage']))
			{
				$perpage = (!empty($args['section']) && $args['section'] == 'admin') ?
					(int)$this->config->adminItemsPerPage : (int)$this->config->frontendItemsPerPage;
			} else
			{
				$perpage = $args['perpage'];
			}

			$page = (int)str_replace('page', '', (!empty($_GET['page']) ? $_GET['page'] : 1));
			$next = ($page - 1) * $perpage + 1;

			$this->itemMapper->alloc((int)$catid);
			$simpleItems = $this->itemMapper->getSimpleItems($args['selector']);

			if(!$simpleItems) return array('items' => $simpleItems, 'total' => 0, 'pagination' => '', 'page' => 1);
			$total = $this->itemMapper->countItems($simpleItems);
			$this->itemMapper->total = $total;
			$simpleItems = $this->itemMapper->filterSimpleItems($args['filter'], $args['order'], $next, $perpage, $simpleItems);
			$pagination = $this->itemMapper->pagination(array(
					'page' => $page,
					'pageurl' => $args['pageurl'],
					'limit' => $perpage)
			);

			return array('items' => $simpleItems, 'total' => $total, 'pagination' => $pagination, 'page' => $page);
		}
	}


	public function itemsMerge($items)
	{
		return array_merge($this->itemMapper->items, $items);
	}


	/**
	 * This method initializes and returns a new or already existing item
	 *
	 * @return \Item
	 */
	public function getItemForEdit($catid, $itemid = 0)
	{
		$editid = ($itemid > 0) ? (int) $itemid : 0;
		if($editid > 0) { $this->itemMapper->limitedInit($catid, $editid); }
		return !empty($this->itemMapper->items[$editid]) ? $this->itemMapper->items[$editid] : new \Item($catid);
	}


	/**
	 * Returns all available SimpleItem objects of the SC categories
	 *
	 * @return mixed array|bool - Category SimpleItem objects array or empty value
	 */
	public function getAllCategories($selector = null)
	{
		$this->itemMapper->alloc($this->config->dummyCategoryId);
		return $this->itemMapper->getSimpleItems($selector);
	}


	/**
	 * Just a simple get item methode
	 *
	 * @param $categoryid
	 * @param $itemid
	 *
	 * @return null | object - SimpleItem object
	 */
	public function getItem($categoryid, $itemid) {
		$this->itemMapper->alloc($categoryid);
		return !empty($this->itemMapper->simpleItems[$itemid]) ? $this->itemMapper->simpleItems[$itemid] : null;
	}


	/**
	 * This method returns raw (unfiltered) array of SimpleItem objects
	 *
	 * @param $categoryid
	 *
	 * @return null | object - SimpleItem object
	 */
	public function getUnfilteredItems($categoryid) {
		$this->itemMapper->alloc($categoryid);
		return !empty($this->itemMapper->simpleItems) ? $this->itemMapper->simpleItems : null;
	}

	/**
	 * This method will called after clicking on Save Item/Category button
	 *
	 * @return bool
	 */
	public function __saveItem($itemType)
	{
		// No Category is selected just return without saving
		if(empty($_POST['categoryid']) && !is_numeric($_POST['categoryid'])) {
			\MsgReporter::setClause('err_save_item_x', array(), true);
			return false;
		}

		// Check if category is valid
		$this->imanager->ProcessCategory();
		if(!$this->imanager->cp->isCategoryValid($_POST['categoryid']))
		{
			\MsgReporter::setClause('err_save_item_c', array(), true);
			return false;
		}

		// Check item name
		$length = strlen(utf8_decode($_POST['name']));
		$itemName = $this->imanager->sanitizer->text($_POST['name']);
		if(empty($itemName))
		{
			\MsgReporter::setClause('err_empty_item_name', array(), true);
			return false;
		}

		$itemId = (!empty((int)@$_GET['edit']) ? (int)$_GET['edit'] : null);

		// This is a dummy category, so check some params and create a real category if necessary
		$realCategory = null;
		if($itemType == 'category')
		{
			if($_POST['categoryid'] != $this->config->dummyCategoryId)
			{
				\MsgReporter::setClause('err_save_item_c', array(), true);
				return false;
			}

			$itemName = $this->imanager->sanitizer->text($itemName,
				array('maxLength' => $this->imanager->config->common->maxcatname)
			);

			if($length > $this->imanager->config->common->maxcatname) {
				\MsgReporter::setClause('err_empty_item_name', array(), true);
			}

			$realCategory = $this->createRealCategory($itemName, $itemId);

			if(!$realCategory) {
				\MsgReporter::setClause('err_create_real_category', array(), true);
				return false;
			}
		}

		$categoryid = $_POST['categoryid'];

		if(!empty($_GET['edit'])) {
			$this->itemMapper->limitedInit($categoryid, (int)$_GET['edit']);
		}

		$curitem = !empty($this->itemMapper->items[(int)@$_GET['edit']]) ?
			$this->itemMapper->items[(int)$_GET['edit']] : new \Item($categoryid);


		if(empty($_POST['slug'])) $newSlug = $this->imanager->sanitizer->pageName($itemName);
		else $newSlug = $this->imanager->sanitizer->pageName($_POST['slug']);

		// Get the old item slug, we should delete it from the routing table
		$oldSlug = !empty($curitem->fields->slug->value) ? $curitem->fields->slug->value : null;

		// Let's go check whether the new slug already exist !empty($oldSlug) &&
		if((string) $newSlug !== (string) $oldSlug) {
			if($this->router->isRouteExists($newSlug)) {
				\MsgReporter::setClause('err_slug_already_exists', array(), true);
				return false;
			}
		}

		// Clean up cached images/files
		$this->imanager->cleanUpCachedFiles($curitem);

		// Check if slug field available
		foreach($curitem->fields as $fieldname => $fieldvalue)
		{
			$fieldinput = !empty($_POST[$fieldname]) ? str_replace('"', "'", $_POST[$fieldname]) : '';
			$inputClassName = '\Input'.ucfirst($fieldvalue->type);
			$InputType = new $inputClassName($curitem->fields->{$fieldname});

			// Check image/file field types
			if($fieldvalue->type == 'fileupload')
			{
				// new item
				if(empty($_GET['edit']) && !empty($_POST['timestamp']))
				{
					// pass temporary image directory
					$tmp_image_dir = IM_IMAGE_UPLOAD_DIR.'tmp_'.(int)$_POST['timestamp'].'_'.$categoryid.'/';
					$fieldinput = $tmp_image_dir;
				} else
				{
					// pass image directory
					$fieldinput = IM_IMAGE_UPLOAD_DIR.$curitem->id.'.'.$categoryid.'/';
				}

				// position is send
				if(isset($_POST['position']) && is_array($_POST['position']))
				{
					$InputType->positions = $_POST['position'];
					$InputType->titles = isset($_POST['title']) ? $_POST['title'] : '';

					if(!file_exists($fieldinput.'config.xml'))
					{
						$xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?><params></params>');
						$i = 0;
						foreach($InputType->positions as $filepos => $filename)
						{
							$xml->image[$i]->name = $filename;
							$xml->image[$i]->position = $filepos;
							$xml->image[$i]->title = !empty($InputType->titles[$filepos])
								? $InputType->titles[$filepos] : '';
							$i++;
						}

					} else
					{
						$xml = simplexml_load_file($fieldinput.'config.xml');
						unset($xml->image);
						$i = 0;
						foreach($InputType->positions as $filepos => $filename)
						{
							$xml->image[$i]->name = $filename;
							$xml->image[$i]->position = $filepos;
							$xml->image[$i]->title = !empty($InputType->titles[$filepos])
								? $InputType->titles[$filepos] : '';
							$i++;
						}
					}

					if(is_dir($fieldinput)) $xml->asXml($fieldinput.'config.xml');
				}
			} elseif($fieldvalue->type == 'password')
			{
				$InputType->confirm = !empty($_POST['password_confirm']) ? $_POST['password_confirm'] : '';
				// refill password field values if empty
				$InputType->password = !empty($curitem->fields->{$fieldname}->value)
					? $curitem->fields->{$fieldname}->value : '';
				$InputType->salt = !empty($curitem->fields->{$fieldname}->salt)
					? $curitem->fields->{$fieldname}->salt : '';
				$fieldinput = !empty($_POST['password']) ? $_POST['password'] : '';
			}

			$resultinput = $InputType->prepareInput($fieldinput);

			if(!isset($resultinput) || empty($resultinput) || is_int($resultinput))
			{
				// parse error codes
				switch ($resultinput)
				{
					case 1:
						\MsgReporter::setClause('err_empty_field_value',
							array('field' => $fieldvalue->label), true);
					case 2:
						\MsgReporter::setClause('err_min_field_value',
							array('field' => $fieldvalue->label, 'num' => $fieldvalue->minimum), true);
					case 3:
						\MsgReporter::setClause('err_max_field_value',
							array('field' => $fieldvalue->label, 'num' => $fieldvalue->maximum), true);
					case 5:
						\MsgReporter::setClause('err_field_value_incompl',
							array('field' => $fieldvalue->label), true);
					case 7:
						\MsgReporter::setClause('err_field_value_match',
							array('field' => $fieldvalue->label), true);
					case 8:
						\MsgReporter::setClause('err_field_value_format',
							array('field' => $fieldvalue->label), true);
					default:
						\MsgReporter::setClause('err_field_value',
							array('field' => $fieldvalue->label), true);
				}
				return false;
			}

			foreach($resultinput as $inputputkey => $inputvalue)
				$curitem->fields->{$fieldname}->{$inputputkey} = $inputvalue;
		}
		// Set some data that should be set automatically
		$curitem->name = $itemName;
		$curitem->position = (!empty($_POST['itempos']) && is_numeric($_POST['itempos'])) ? $_POST['itempos'] : null;
		if($itemType == 'category' && !$curitem->id && !$this->config->catAutoActivate) $curitem->active = 0;
		elseif($itemType == 'category' && !$curitem->id && $this->config->catAutoActivate) $curitem->active = 1;
		elseif($itemType == 'item' && !$curitem->id && !$this->config->itemAutoActivate) $curitem->active = 0;
		elseif($itemType == 'item' && !$curitem->id && $this->config->itemAutoActivate) $curitem->active = 1;
		$curitem->setFieldValue('slug', $newSlug);

		// OK, let's save the item
		$type = 2;
		if($realCategory) {
			$curitem->id = $realCategory->id;
			$type = 1;
			if(!$curitem->forcedSave()) {
				\MsgReporter::setClause('err_save_item_x', array(), true);
				return false;
			}
		} elseif(!$realCategory && !$curitem->save()) {
			\MsgReporter::setClause('err_save_item_x', array(), true);
			return false;
		}
		// Delete old route and set a new one
		if(!empty($oldSlug)) {
			$this->router->removeRoute($oldSlug);
		}
		$this->router->setRoute($curitem, $type);

		// Just expire all markup caches
		$this->imanager->getSectionCache()->expire();

		// Check if it's a new item, so temporary image directory should be renamed
		if(!empty($tmp_image_dir) && file_exists($tmp_image_dir))
		{
			$this->imanager->renameTmpDir($curitem);
			// clean up directories from orphaned data
			$this->imanager->cleanUpTempContainers('imageupload');
			$this->imanager->cleanUpTempContainers('fileupload');
		}
		// Save SimpleItem object now
		if($this->itemMapper->alloc($curitem->categoryid) !== true)
		{
			$this->itemMapper->init($categoryid);
			if(!empty($this->itemMapper->items))
			{
				$this->itemMapper->simplifyBunch($this->itemMapper->items);
				$this->itemMapper->save();
			}
		}
		$this->itemMapper->simplify($curitem);
		$this->itemMapper->save();
		// Set the IndexData
		if($realCategory) {$this->setTotalCategories();}
		else {$this->pushRecentItems($curitem); $this->setTotalItems();}

		\MsgReporter::setClause('success_item_save');
		// Jaaa das ist das Beste! ;-)
		if(empty($_GET['edit'])) $_GET['edit'] = $curitem->id;

		return true;
	}


	/**
	 * The methd to delete real- and dummy- categories
	 *
	 * @return bool
	 */
	public function __deleteCategory()
	{
		$id = (int)$_GET['delete'];
		$this->itemMapper->alloc($this->config->dummyCategoryId);
		$item = (!empty($this->itemMapper->simpleItems[$id]) ? $this->itemMapper->simpleItems[$id] : null);
		if($this->deleteItem($this->config->dummyCategoryId, $id))
		{
			$this->router->removeRoute($item->slug);
			if($realCategory = $this->getRealCategoryById($id)) {
				$this->imanager->deleteCategory($realCategory->id);
				// todo: fix it!
				\MsgReporter::removeClauseByValue(
					'<li class="notify" style="display: list-item;">{im_simple_catalog/category_deleted}</li>');
			}
			$this->setTotalCategories();
			unset($_SESSION['scdata']['curcat']);
			return true;
		}
		return false;
	}


	/**
	 * The methd deletes a product item
	 *
	 * @return bool
	 */
	public function __deleteItem()
	{
		$id = (int)$_GET['delete'];
		$curcatid = $this->getCurrentCategoryId();
		$this->itemMapper->alloc($curcatid);
		$item = (!empty($this->itemMapper->simpleItems[$id]) ? $this->itemMapper->simpleItems[$id] : null);
		if($this->deleteItem($curcatid, $id)) {
			$this->router->removeRoute($item->slug);
			$this->setTotalItems();
			if(is_null($this->dashboardData)) $this->dashboardData = $this->getDashboardData();
			// is deleted item inside recent items array?
			foreach($this->dashboardData->recentItems as $key => $recentItems) {
				if($recentItems['id'] == $id && $recentItems['categoryid'] == $curcatid) {
					unset($this->dashboardData->recentItems[$key]);
				}
			}
			return true;
		}
		return false;
	}


	/**
	 * Delete a single item
	 *
	 * @return bool
	 */
	private function deleteItem($catid, $itemid)
	{
		if(!empty($catid) && !empty($itemid)) return $this->imanager->deleteItem($itemid, $catid);
	}


	/**
	 * Activate / Deactivate category items
	 *
	 * @return bool|void
	 *
	 */
	public function __toggleCategory()
	{
		if(empty($_GET['activate'])) return;
		$id = (int)$_GET['activate'];
		$this->itemMapper->limitedInit($this->config->dummyCategoryId, $id);
		return !empty($this->itemMapper->items[$id]) ? $this->toggle($this->itemMapper->items[$id]) : false;
	}


	/**
	 * Activate / Deactivate product items
	 *
	 * @return bool|void
	 *
	 */
	public function __toggleItem()
	{
		if(empty($_GET['activate'])) return;
		$id = (int)$_GET['activate'];
		$this->itemMapper->limitedInit($this->getCurrentCategoryId(), $id);
		return !empty($this->itemMapper->items[$id]) ? $this->toggle($this->itemMapper->items[$id]) : false;
	}


	/**
	 * Toggle item
	 *
	 * @param \Item $item
	 *
	 * @return bool
	 */
	private function toggle(\Item $item)
	{
		$item->active = ((int)$item->active == 1) ? 0 : 1;

		if($item->save())
		{
			if($this->itemMapper->alloc($item->categoryid) !== true)
			{
				$this->itemMapper->init($item->categoryid);
				if(!empty($this->itemMapper->items))
				{
					$this->itemMapper->simplifyBunch($this->itemMapper->items);
					$this->itemMapper->save();
				}
			}
			$this->itemMapper->simplify($item);
			$this->itemMapper->save();
			return true;
		}
	}


	/**
	 * Process category list: changes items positions
	 *
	 * @param array $args
	 */
	public function __processCategoryList($args = array(
			'type' => 'simple_items',
			'section' => 'admin'
		)
	){
		ob_start();
		if($args['type'] == 'simple_items')
		{
			$positions = $_POST['positions'];
			usort($positions, array('\ImCatalog\Util', 'sortArray'));
			$last = end($positions);
			$this->itemMapper->limitedInit($this->config->dummyCategoryId, $positions[0]['id'], $last['id']);

			if(empty($this->itemMapper->items)) return;

			foreach($positions as $key => $val)
			{
				if(empty($this->itemMapper->items[$val['id']])) continue;
				// When item position is changed
				if($this->itemMapper->items[$val['id']]->position != $val['position'])
				{
					$this->itemMapper->items[$val['id']]->position = (int)$val['position'];

					if(!$this->itemMapper->items[$val['id']]->save()) continue;

					if($this->itemMapper->alloc($this->itemMapper->items[$val['id']]->categoryid) !== true)
					{
						$this->itemMapper->init($this->itemMapper->items[$val['id']]->categoryid);
						if(!empty($this->itemMapper->items))
						{
							$this->itemMapper->simplifyBunch($this->itemMapper->items);
							$this->itemMapper->save();
						}
					}
					$this->itemMapper->simplify($this->itemMapper->items[$val['id']]);
					$this->itemMapper->save();
				}
			}
		}
		$o = ob_get_contents();
		if(!empty($o)) \Util::dataLog($o);
		if($this->config->debug) echo $o;
		ob_end_clean();
	}


	/**
	 * Process item list: changes items positions and items category
	 *
	 * @param array $args
	 */
	public function __processItemList($args = array(
		'type' => 'simple_items',
		'section' => 'admin'
	)
	){
		ob_start();
		if($args['type'] == 'simple_items')
		{
			if(!empty($_GET['scitemlist']))
			{
				$id = (int)$_GET['scitemlist'];
				$this->itemMapper->alloc($this->config->dummyCategoryId);
				$simpleItems = $this->itemMapper->getSimpleItems();
				if(!empty($simpleItems[$id])) {
					$this->setCurrentCategory($simpleItems[$id]);
				}
			}
			$currentCategoryId = $this->getCurrentCategoryId();

			if(empty($_POST['positions'])) return;
			$positions = $_POST['positions'];
			usort($positions, array('\ImCatalog\Util', 'sortArray'));
			$last = end($positions);
			$this->itemMapper->limitedInit($currentCategoryId, $positions[0]['id'], $last['id']);

			if(empty($this->itemMapper->items)) return;

			foreach($positions as $key => $val)
			{
				if(empty($this->itemMapper->items[$val['id']])) continue;
				// When item position is changed
				if($this->itemMapper->items[$val['id']]->position != $val['position'])
				{
					$this->itemMapper->items[$val['id']]->position = (int)$val['position'];

					if(!$this->itemMapper->items[$val['id']]->save()) continue;

					if($this->itemMapper->alloc($this->itemMapper->items[$val['id']]->categoryid) !== true)
					{
						$this->itemMapper->init($this->itemMapper->items[$val['id']]->categoryid);
						if(!empty($this->itemMapper->items))
						{
							$this->itemMapper->simplifyBunch($this->itemMapper->items);
							$this->itemMapper->save();
						}
					}
					$this->itemMapper->simplify($this->itemMapper->items[$val['id']]);
					$this->itemMapper->save();
				}
			}
		}
		$o = ob_get_contents();
		if(!empty($o)) \Util::dataLog($o);
		if($this->config->debug) echo $o;
		ob_end_clean();
	}


	/**
	 * This method returns a current SC-category id.
	 * If the session variable does not exists yet, it now will be created
	 *
	 * @return bool
	 */
	public function getCurrentCategoryId()
	{
		if(empty($_SESSION['scdata']['curcat']['id']))
		{
			$simpleItems = $this->getAllCategories();
			if(!$simpleItems)
			{
				\MsgReporter::setClause('note_no_one_cat_found');
				return false;
			}
			$this->setCurrentCategory(current($simpleItems));
		}

		return $_SESSION['scdata']['curcat']['id'];
	}

	/**
	 * Creates new current category session
	 *
	 * @param \SimpleItem $category
	 */
	private function setCurrentCategory(\SimpleItem $category)
	{
		$_SESSION['scdata']['curcat']['id'] = $category->id;
		$_SESSION['scdata']['curcat']['name'] = $category->name;
		$_SESSION['scdata']['curcat']['slug'] = $category->slug;
	}


	private function getRealCategoryById($id)
	{
		return $this->imanager->getCategoryMapper()->getCategory((int)$id);
	}


	private function getRealCategoryByName($name)
	{
		return $this->imanager->getCategoryMapper()->getCategory('name='.$this->imanager->sanitizer->text($name));
	}


	/**
	 * This function creates or updates the real product categories
	 * We'll need to check whether the category name already exists to
	 * avoid overwriting already existing non-SC categories.
	 *
	 * @return bool|object - Category object or false on error
	 */
	private function createRealCategory($name, $id = null)
	{
		if(!is_null($id)) $realCategory = $this->getRealCategoryById($id);
		else $realCategory = $this->getRealCategoryByName($name);

		if($realCategory) {
			// Change the category name, it can be occurred when selected by id
			if($realCategory->name !== $name) {
				$realCategory->name = $this->imanager->sanitizer->text($name);
				$realCategory->slug = $this->imanager->sanitizer->pageName($name);
				$realCategory->save();
			}
			return $realCategory;
		}
		// Now, verify whether the name duplicate of the category exists
		if(!is_null($id)) {
			// Let's check for orphaned dummy's
			if(!$realCategory) {
				\MsgReporter::setClause('err_orphaned_category', array(), true);
				return false;
			}
			$realCategory = $this->getRealCategoryByName($name);
			if($realCategory) {
				\MsgReporter::setClause('err_real_category_name_exists', array(), true);
				return false;
			}
		}
		// OK, the product category doesn't exist, let's go ahead and create a new one
		// Use ItemManager's language file
		\MsgReporter::$dir = 'imanager';
		if($this->imanager->createCategoryByName($name, true) !== true) return false;

		$realCategory = $this->getRealCategoryByName($name);
		if(!$realCategory) return false;

		// Include default fields and field data of product category
		if(file_exists($this->config->pluginRoot.'/inc/custom.prodcatdata.php')) {
			include($this->config->pluginRoot.'/inc/custom.prodcatdata.php');
		} else {
			include($this->config->pluginRoot.'/inc/prodcatdata.php');
		}
		// Create fields
		if($this->imanager->createFields($fields) !== true) {
			$this->imanager->deleteCategory($realCategory->id);
			return false;
		}
		// Set field data
		if($this->setFiedData($realCategory->id, $fieldsdata) !== true) {
			$this->imanager->deleteCategory($realCategory->id);
			return false;
		}
		// Switch back to the SimpleCatalog language file
		\MsgReporter::$dir = $this->config->pluginId;

		return $realCategory;
	}


	/**
	 * This method can be used to create/extend the field data of a category
	 *
	 * @param $catid - The category id that you want to update
	 * @param array $fieldsdata - The data to update
	 *
	 * @return bool
	 */
	private function setFiedData($catid, array $fieldsdata)
	{
		$cf = new \FieldMapper();
		$cf->init($catid);

		foreach($fieldsdata as $input)
		{
			// Field already exists
			$currfield = $cf->getField((int)$input['field']);

			if(!$currfield)
			{
				\MsgReporter::setClause('err_field_id', array(), true);
				return false;
			}

			$currfield->default = !empty($input['default']) ? str_replace('"', "'",
				$input['default']) : '';
			$currfield->info = !empty($input['info']) ? str_replace('"', "'", $input['info']) : '';
			$currfield->required = (isset($input['required']) && $input['required'] > 0) ? 1 : null;
			$currfield->minimum = (isset($input['min_field_input']) && intval($input['min_field_input']) > 0)
				? intval($input['min_field_input']) : null;
			$currfield->maximum = (isset($input['max_field_input']) && intval($input['max_field_input']) > 0)
				? intval($input['max_field_input']) : null;
			$currfield->areaclass = !empty($input['areaclass']) ? str_replace('"', "'",
				$input['areaclass']) : '';
			$currfield->labelclass = !empty($input['labelclass']) ? str_replace('"', "'",
				$input['labelclass']) : '';
			$currfield->fieldclass = !empty($input['fieldclass']) ? str_replace('"', "'",
				$input['fieldclass']) : '';

			// process custom Fieldtype settings
			foreach($input as $key => $value)
			{
				if(strpos($key, 'custom-') !== false)
				{
					$fieldkey = str_replace('custom-', '', $key);
					$currfield->configs->$fieldkey = $value;
				}
			}
			if(!$currfield->save()) return false;
		}
		return true;
	}


	/**
	 * This function returns DashboardData object
	 *
	 * @return bool|object - DashboardData object
	 */
	public function getDashboardData()
	{
		if(is_null($this->dashboardData))
		{
			$this->itemMapper->alloc($this->config->indexDataId);
			if(empty($this->itemMapper->simpleItems[$this->config->indexDataDashboardId])) {
				\MsgReporter::setClause('err_dashboard_data_not_found', array(), true);
				return false;
			}
			$this->dashboardData = unserialize(base64_decode(
				$this->itemMapper->simpleItems[$this->config->indexDataDashboardId]->data));
		}
		return $this->dashboardData;
	}

	public function setDashboardData($name, $value)
	{
		if(is_null($this->dashboardData))
		{
			if(($this->dashboardData = $this->getDashboardData()) === false) return false;
		}
		$this->dashboardData->{$name} = $value;
	}


	/**
	 * A Methode to save DashboardData object
	 *
	 * @return bool|void
	 */
	public function saveDashboardData()
	{
		if(is_null($this->dashboardData)) {
			if(($this->dashboardData = $this->getDashboardData()) === false) return false;
		}
		$this->itemMapper->init($this->config->indexDataId);

		$item = $this->itemMapper->getItem($this->config->indexDataDashboardId);
		if($item === false) {
			\MsgReporter::setClause('err_saving_indexdata', array(), true);
			return false;
		}
		$value = base64_encode(serialize($this->dashboardData));
		$item->setFieldValue('data', $value, false);
		$item->save();
		if($this->itemMapper->alloc($this->config->indexDataId) !== true)
		{
			$this->itemMapper->init($this->config->indexDataId);
			if(!empty($this->itemMapper->items))
			{
				$this->itemMapper->simplifyBunch($this->itemMapper->items);
				$this->itemMapper->save();
			}
		}
		$this->itemMapper->simplify($item);
		$this->itemMapper->save();
	}


	/**
	 * We use this method to count SimpleCatalog categories
	 * the value will save automatically to our dashboardData - DataIndex
	 *
	 * @return void
	 */
	public function setTotalCategories()
	{
		$categories = $this->getAllCategories();
		$this->setDashboardData('categoriesTotal', @count($categories));
		$this->saveDashboardData();
	}


	/**
	 * We use this method to count the items of all our SimpleCatalog categories
	 * The value will save automatically to our dashboardData - DataIndex
	 *
	 * @return void
	 */
	public function setTotalItems()
	{
		$categories = $this->getAllCategories();
		if(!$categories) return;
		$total = 0;
		foreach($categories as $category)
		{
			$this->itemMapper->alloc($category->id);
			$total += (int)$this->itemMapper->total;
		}
		$this->setDashboardData('itemsTotal', $total);
		$this->saveDashboardData();
	}


	/**
	 * We use this method to insert a new item to our recent items array
	 * The value will save automatically to our dashboardData - DataIndex
	 *
	 * @param \Item $item
	 *
	 * @return bool|void
	 */
	public function pushRecentItems(\Item $item)
	{
		if(is_null($this->dashboardData)) {
			if(($this->dashboardData = $this->getDashboardData()) === false) return false;
		}
		// it's a new item or just an update?
		$existkey = null;
		foreach($this->dashboardData->recentItems as $key => $recentItems) {
			if($recentItems['id'] == $item->id && $recentItems['categoryid'] == $item->categoryid) {$existkey = $key;}
		}
		$url = '';
		if(!empty($item->fields->image->fullurl[0])) {
			@unlink($item->fields->image->path[0].'thumbnail/300x300_'.$item->fields->image->file_name[0]);
			//if(!file_exists($item->fields->image->path[0].'thumbnail/400x300_'.$item->fields->image->file_name[0]))
			//{
				$thumb = \PhpThumbFactory::create($item->fields->image->fullpath[0]);
				$path_parts = pathinfo($item->fields->image->fullpath[0]);
				$thumb->adaptiveResize(300, 300);
				$thumb->save($item->fields->image->path[0].'thumbnail/300x300_'.
					$item->fields->image->file_name[0], $path_parts['extension']);
			//}
			$url = $item->fields->image->url[0].'thumbnail/300x300_'.$item->fields->image->file_name[0];
		}
		if(is_null($existkey))
		{
			array_unshift($this->dashboardData->recentItems, array(
					'id' => $item->id,
					'categoryid' => $item->categoryid,
					'src' => $url,
					'title' => $item->name
				)
			);
		} else
		{
			$this->dashboardData->recentItems[$existkey]['src'] = $url;
			$this->dashboardData->recentItems[$existkey]['title'] = $item->name;
		}
		$this->dashboardData->recentItems = array_slice($this->dashboardData->recentItems, 0, 4);
		$this->saveDashboardData();
	}
}